(function(ViewController, utils, log) {
    'use strict';

    var retryDomains = [
            'yahoo.com',
            'ymail.com'
        ],
        disableImageProgressFeature = false,
        regexDomainPattern = '^https?\:\/\/[^\/|\?|\:]*\.';

    function ImageViewController() {
        ViewController.call(this);

        this._retryCallback = function() {
            var url = this.el.getAttribute('src'),
                imgContainer,
                shouldRetry;

            // remove listener. only allow single retry
            this.removeAllListeners();

            shouldRetry = utils.findFirst( retryDomains, function(domain) {
                var regex = new RegExp(regexDomainPattern + domain, 'i');

                return regex.test(url);
            });

            if (shouldRetry) {
                // force reload
                this.retry();
            } else {
                imgContainer = getImgContainer(this.el);
                // If loading fails, add the pending class
                if (imgContainer) {
                    imgContainer.classList.add(IMG_PENDING_CLASS);
                }
            }
        }.bind(this);

        this.removeAllListeners = function() {
            this.el.removeEventListener('error', this._retryCallback);
            this.el.removeEventListener('load', this._onLoadCallback);
        }.bind(this);

        this._onLoadCallback = function() {
            var imgContainer = getImgContainer(this.el);

            // Load success, remove pending class
            if (imgContainer) {
                imgContainer.classList.remove(IMG_PENDING_CLASS);
            }

            this.removeAllListeners();
        }.bind(this);
    }

    ImageViewController.prototype = Object.create(ViewController.prototype);
    ImageViewController.prototype.constructor = ImageViewController;

    /**
     * Attempts to refetch the image from the src
     */
    ImageViewController.prototype.retry = function() {
        var url = this.el.getAttribute('src');

        // Only retry if the img haven't been retried yet
        if (this.el.className.indexOf(IMG_RETRIED_CLASS) < 0) {
            log.d('retrying url: ' + url);

            // retry by set the src to the same value
            this.el.setAttribute('src', url);
            this.el.classList.add(IMG_RETRIED_CLASS);
        }

    };

    /**
     * Attach to an existing Attachment Card in the DOM
     *
     * @param {HtmlElement} imgEl
     * @param {HtmlElement} removeBtnEl
     * @override
     */
    ImageViewController.prototype.attach = function(imgEl) {
        var imgContainer = getImgContainer(imgEl),
            parentNode = imgEl.parentNode,
            inlineImgClassPrefix = INLINE_IMG_CLASS_NAME_PREFIX,
            isCompose = typeof COMPOSE_CONTENT_ELEMENT_ID !== "undefined";

        ViewController.prototype.attach.call(this, imgEl);

        // If it is compose, add the img wrapper container
        if (isCompose && imgEl.className && imgEl.className.indexOf(inlineImgClassPrefix) > 0 &&
            parentNode.className.indexOf(INLINED_ATTACHMENT_CONTAINER_CLASS_NAME) < 0) {
            imgContainer = document.createElement("div");

            imgEl.style.width = "100%";
            imgContainer.classList.add(INLINED_ATTACHMENT_CONTAINER_CLASS_NAME);
            parentNode.insertBefore(imgContainer, imgEl);
            imgContainer.appendChild(imgEl);
        } else if (parentNode.className.indexOf(INLINED_ATTACHMENT_CONTAINER_CLASS_NAME) > 0 &&
            parentNode.classList && parentNode.classList.contains(INLINED_ATTACHMENT_CONTAINER_CLASS_NAME) < 0) {
            // A special case that our class name get modified by other client like append yiv prefix.
            // We need to add it back for css to working correctly
            parentNode.classList.add(INLINED_ATTACHMENT_CONTAINER_CLASS_NAME);
        }

        // add loader for http/https req. (not content:// etc ...)
        // Note 'startsWith' is not defined in Android 4.4
        // image threshold, either width or height should be min 200px.
        // do not add spinner, if the image height or width is not specified
        if (disableImageProgressFeature && imgEl.src && imgEl.src.substr(0, HTTP_SCHEME.length) === HTTP_SCHEME) {
            if (imgEl.height > 200 || imgEl.width > 200) {
                // image loader element
                imgContainer = document.createElement("div");
                imgContainer.classList.add('image-loader');
                imgContainer.id = imgEl.id;

                // imgWrapper is to wrap img and image-loader
                var imgWrapper = document.createElement('div');
                imgWrapper.setAttribute("style", "position: relative;");

                // set wrapper as child (instead of the element)
                parentNode.replaceChild(imgWrapper, imgEl);
                // set element as child of wrapper
                imgWrapper.appendChild(imgEl);
                // add loader element
                imgWrapper.appendChild(imgContainer);
            }
        }

        imgContainer = getImgContainer(imgEl);

        if (imgContainer) {
            imgContainer.classList.add(IMG_PENDING_CLASS);
            imgContainer.setAttribute("contenteditable", false);
        }

        this.el.addEventListener('error', this._retryCallback);
        this.el.addEventListener('load', this._onLoadCallback);
    };

    /**
     * Render image into current DOM, we need to insert the whole card instead of just image.
     * @param {HtmlElement} imgEl image node we want to process
     */
    ImageViewController.prototype.renderImage = function(imgEl) {
        var imgContainer = getImgContainer(imgEl);
        ViewController.prototype.renderIntoSelection.call(this, true, imgContainer);
    };

    /**
     * Get card container for current image/gif
     *
     * @param {HtmlElement} imgEl
     * @return {HtmlElement} card container of current img if exists
     */
    function getCardContainer(imgEl) {
        var imgContainer = imgEl.parentNode;

        while (imgContainer) {
            if (imgContainer.className &&
                    (imgContainer.className.indexOf(CARD_EL_CLASS_NAME) >= 0 ||
                    imgContainer.className.indexOf(UNIFIED_IMG_EL_CLASS_NAME) >= 0)) {
                return imgContainer;
            } else {
                imgContainer = imgContainer.parentNode;
            }
        }

        return null;
    }

    /**
     * Get the container of the img. If is a img card, return the card, otherwise return the immediate parent node for inline image.
     * @param {HtmlElement} image node we want to process
     */
    function getImgContainer(imgEl) {
        var imgContainer = getCardContainer(imgEl);

        if (!imgContainer && imgEl.className.indexOf(INLINE_IMG_CLASS_NAME_PREFIX) >= 0) {
            imgContainer = imgEl.parentNode;
        }

        return imgContainer;
    }

    ImageViewController.prototype.getCardContainer = getCardContainer;

    // export
    window.ImageViewController = ImageViewController;
})(window.ViewController, window.utils, window.log);
